import org.codehaus.groovy.grails.plugins.springsecurity.Secured
import org.codehaus.groovy.grails.commons.ConfigurationHolder

@Secured(['ROLE_AppAdmin', 'ROLE_Manager', 'ROLE_AssetManager'])
class AssetDetailedController extends BaseController {

    def csvService
    def filterService
    def exportService
    def assetService

    // the delete, save and update actions only accept POST requests
    static allowedMethods = [delete:'POST', save:'POST', update:'POST', saveCopy:'POST']

    @Secured(['ROLE_AppAdmin', 'ROLE_Manager', 'ROLE_AssetManager', 'ROLE_AssetUser'])
    def index = { redirect(action:search,params:params) }

    def importAssetTree = {
    }

    def importAssetTreeSave = {
        def result = csvService.importAssetTree(request)

        if(!result.error)
            flash.message = g.message(code: "asset.tree.import.success")
        else
            flash.errorMessage = g.message(code: result.error.code, args: result.error.args)

        redirect(action: importAssetTree)
    }

    /**
    * Export a csv template.
    * NOTE: IE has a 'validating' bug in dev mode that causes the export to take a long time!
    * This does not appear to be a problem once deployed to Tomcat.
    */
    @Secured(['ROLE_AppAdmin', 'ROLE_Manager', 'ROLE_AssetManager', 'ROLE_AssetUser'])
    def exportAssetTreeTemplate = {
        response.contentType = ConfigurationHolder.config.grails.mime.types["csv"]
        response.setHeader("Content-disposition", "attachment; filename=AssetTreeTemplate.csv")
        def s = csvService.buildAssetTreeTemplate()
        render s
    }

    def exportAssetTreeTest = {
        response.contentType = ConfigurationHolder.config.grails.mime.types["csv"]
        response.setHeader("Content-disposition", "attachment; filename=AssetTreeTestFile.csv")
        def s = csvService.buildAssetTreeTest()
        render s
    }

    /**
    * Export the entire asset tree.
    */
    @Secured(['ROLE_AppAdmin', 'ROLE_Manager', 'ROLE_AssetManager', 'ROLE_AssetUser'])
    def exportAssetTree = {

        def assetList = Asset.list()

        response.contentType = ConfigurationHolder.config.grails.mime.types["csv"]
        response.setHeader("Content-disposition", "attachment; filename=AssetTree.csv")
        def s = csvService.buildAssetTree(assetList)
        render s
    }

    /**
    * List action.
    */
    @Secured(['ROLE_AppAdmin', 'ROLE_Manager', 'ROLE_AssetManager', 'ROLE_AssetUser'])
    def list = {
        params.max = Math.min( params.max ? params.max.toInteger() : 10,  100)
        [ assetInstanceList: Asset.list( params ), assetInstanceTotal: Asset.count() ]
    }

    /**
    * Search action.
    */
    @Secured(['ROLE_AppAdmin', 'ROLE_Manager', 'ROLE_AssetManager', 'ROLE_AssetUser'])
    def search = {

        if(session.taskSearchParamsMax)
            params.max = session.taskSearchParamsMax

        params.max = Math.min( params.max ? params.max.toInteger() : 10,  1000)

        def assetInstanceList = []
        def assetInstanceTotal
        def filterParams = [:]

        // Quick Search:
        if(!params.filter) {
            assetInstanceList = Asset.list( params )
            assetInstanceTotal = Asset.count()
            filterParams.quickSearch = params.quickSearch
        }
        else {
        // filterPane:
            assetInstanceList = filterService.filter( params, Asset )
            assetInstanceTotal = filterService.count( params, Asset )
            filterParams = com.zeddware.grails.plugins.filterpane.FilterUtils.extractFilterParams(params)
        }

        // export plugin:
        if(params?.format && params.format != "html") {

            def dateFmt = { date ->
                formatDate(format: "EEE, dd-MMM-yyyy", date: date)
            }

//             def fmtAsset = { m ->
//                     def r = ''
//                     def assetInstance = Asset.findByName(m)
// 
//                     r +=  assetInstance
//                     r += ", "
// 
//                     def  lastSubAsset = assetInstance.subAssets.size() - 1
//                     assetInstance.subAssets.eachWithIndex() { obj, i ->
//                         r += "\"" + obj + "\""
//                         if( i < lastSubAsset )
//                             r += ", "
//                     }
//                     return r
//             }

//             def fmtSubAsset = { m ->
//                     def r = ''
//                     m.each() {
//                         def machine = Machine.findByName(it)
//                         def assemblies = machine.assemblies
//                         r += machine.name
//                         r += " "
//                         r += assemblies
//                         r += " "
//                     }
//                     return r
//             }

            String title = "Asset List."

            response.contentType = ConfigurationHolder.config.grails.mime.types[params.format]
            response.setHeader("Content-disposition", "attachment; filename=tasks.${params.extension}")
            List fields = ["section.site",
                                "section",
                                "name",
                                "description"]
            Map labels = ["section.site": "Site",
                                "section": "Section",
                                "name": "Asset",
                                "description": "Description"]
//             Map labels
//             Map formatters = ["subAsset.name": fmtSubAsset]
            Map formatters = [:]
            Map parameters = [title: title, separator: ","]

            exportService.export(params.format,
                                                response.outputStream,
                                                assetInstanceList.sort { p1, p2 -> p1.name.compareToIgnoreCase(p2.name) },
                                                fields,
                                                labels,
                                                formatters,
                                                parameters)
        }

        // Add some basic params to filterParams.
        filterParams.max = params.max
        filterParams.offset = params.offset?.toInteger() ?: 0
        filterParams.sort = params.sort ?: "id"
        filterParams.order = params.order ?: "desc"

        return[ assetInstanceList: assetInstanceList,
                assetInstanceTotal: assetInstanceTotal,
                filterParams: filterParams ]

    } // end search()

    /**
    * Show action.
    */
    @Secured(['ROLE_AppAdmin', 'ROLE_Manager', 'ROLE_AssetManager', 'ROLE_AssetUser'])
    def show = {

        // In the case of an actionSubmit button, rewrite action name from 'index'.
        if(params._action_Show)
        { params.action='show' }

        def assetInstance = Asset.get( params.id )

        if(!assetInstance) {
            flash.message = "Asset not found with id ${params.id}"
            redirect(action:search)
        }
        else { return [ assetInstance : assetInstance ] }
    }

    /**
    * Delete action.
    */
    def delete = {
        def assetInstance = Asset.get( params.id )
        if(assetInstance) {
            try {
                assetInstance.delete(flush:true)
                flash.message = "Asset ${params.id} deleted"
                redirect(action:search)
            }
            catch(org.springframework.dao.DataIntegrityViolationException e) {
                flash.message = "Asset ${params.id} could not be deleted"
                redirect(action:show,id:params.id)
            }
        }
        else {
            flash.message = "Asset not found with id ${params.id}"
            redirect(action:search)
        }
    }

    /**
    * Edit action.
    */
    def edit = {

        // In the case of an actionSubmit button, rewrite action name from 'index'.
        if(params._action_Edit)
        { params.action='edit' }

        def assetInstance = Asset.get( params.id )

        if(!assetInstance) {
            flash.message = "Asset not found with id ${params.id}"
            redirect(action:search)
        }
        else {
            return [ assetInstance : assetInstance, possibleAssetSubItems: assetService.possibleAssetSubItems() ]
        }
    }

    /**
    * Update action.
    */
    def update = {
        def assetInstance = Asset.get( params.id )
        if(assetInstance) {
            if(params.version) {
                def version = params.version.toLong()
                if(assetInstance.version > version) {

                    assetInstance.errors.rejectValue("version", "asset.optimistic.locking.failure", "Another user has updated this Asset while you were editing.")
                    render(view:'edit',model:[assetInstance:assetInstance, possibleAssetSubItems: assetService.possibleAssetSubItems()])
                    return
                }
            }

            assetInstance.properties = params

            assetInstance.setAssetSubItemsFromCheckBoxList(params.assetSubItems)

            if(!assetInstance.hasErrors() && assetInstance.save(flush: true)) {
                flash.message = "Asset ${params.id} updated"
                redirect(action:show,id:assetInstance.id)
            }
            else {
                render(view:'edit',model:[assetInstance:assetInstance, possibleAssetSubItems: assetService.possibleAssetSubItems()])
            }
        }
        else {
            flash.message = "Asset not found with id ${params.id}"
            redirect(action:list)
        }
    }

    /**
    * Create action.
    */
    def create = {
        def result = assetService.create(params)

        if(!result.error)
            return [assetInstance: result.assetInstance]

        flash.errorMessage = g.message(code: result.error.code, args: result.error.args)
        redirect(action: search)
    }

    /**
    * Copy action.
    */
    def copy = {
        def result = assetService.copy(params)

        if(!result.error)
            return [assetInstance: result.assetInstance, assetToCopy: result.assetToCopy]

        flash.errorMessage = g.message(code: result.error.code, args: result.error.args)
        redirect(action: search)
    }

    /**
    * Save action.
    */
    def save = {
        def result = assetService.save(params)

        if(!result.error) {
            flash.message = g.message(code: "default.create.success", args: ["Asset", result.assetInstance.id])
            redirect(action:show, id: result.assetInstance.id)
            return
        }

        render(view:'create', model:[assetInstance: result.assetInstance])
    }

    /**
    * Copy save action.
    */
    def saveCopy = {
        def result = assetService.saveCopy(params)

        if(!result.error) {
            flash.message = g.message(code: "default.create.success", args: ["Asset", result.assetInstance.id])
            redirect(action:show, id: result.assetInstance.id)
            return
        }

        if(result.error.code == "default.not.found") {
            flash.message = g.message(code: result.error.code, args: ["Asset", params.assetToCopy?.id])
            redirect(action: list)
            return
        }

        if(result.error.code != "default.create.failure") {
            flash.errorMessage = g.message(code: result.error.code, args: ["Asset"])
        }

        render(view:'copy', model:[assetInstance: result.assetInstance, assetToCopy: result.assetToCopy])
    }

} // end class
